var __extends = this.__extends || function (d, b) {
    function __() { this.constructor = d; }
    __.prototype = b.prototype;
    d.prototype = new __();
};
var Microsoft;
(function (Microsoft) {
    (function (VisualStudio) {
        (function (DiagnosticsHub) {
            "use strict";
            var QpcTimeProperties = (function () {
                function QpcTimeProperties(collectionStartTime, frequency) {
                    this._collectionStartTime = collectionStartTime;
                    this._frequency = frequency;
                }
                QpcTimeProperties.prototype.getCollectionStartTime = function () {
                    return this._collectionStartTime;
                };
                QpcTimeProperties.prototype.getFrequency = function () {
                    return this._frequency;
                };
                QpcTimeProperties.prototype.convertQpcTimestampToMilliseconds = function (qpcValue) {
                    var twoToThe64 = 18446744073709552000;
                    var originalQpcValue = qpcValue;
                    if(qpcValue < 0) {
                        qpcValue += twoToThe64;
                    }
                    if(qpcValue < 0 || qpcValue > twoToThe64) {
                        throw new Error("Invalid qpcValue:" + originalQpcValue);
                    }
                    var ticks = qpcValue - this._collectionStartTime;
                    if(ticks < 0) {
                        ticks += twoToThe64;
                    }
                    var seconds = ticks / this._frequency;
                    var milliseconds = seconds * 1000;
                    return milliseconds;
                };
                QpcTimeProperties.prototype.convertMillisecondsToQpcTimestamp = function (milliseconds) {
                    var twoToThe64 = 18446744073709552000;
                    if(milliseconds < 0) {
                        throw new Error("Invalid milliseconds:" + milliseconds);
                    }
                    var seconds = milliseconds / 1000;
                    var ticks = seconds * this._frequency;
                    var qpcValue = ticks + this._collectionStartTime;
                    if(qpcValue >= twoToThe64) {
                        qpcValue -= twoToThe64;
                    }
                    return qpcValue;
                };
                return QpcTimeProperties;
            })();
            DiagnosticsHub.QpcTimeProperties = QpcTimeProperties;            
            (function (Controls) {
                (function (Charting) {
                    (function (GraphScaleType) {
                        GraphScaleType._map = [];
                        GraphScaleType._map[0] = "Unknown";
                        GraphScaleType.Unknown = 0;
                        GraphScaleType._map[1] = "Fixed";
                        GraphScaleType.Fixed = 1;
                        GraphScaleType._map[2] = "Dynamic";
                        GraphScaleType.Dynamic = 2;
                        GraphScaleType._map[3] = "Expanding";
                        GraphScaleType.Expanding = 3;
                    })(Charting.GraphScaleType || (Charting.GraphScaleType = {}));
                    var GraphScaleType = Charting.GraphScaleType;
                    (function (ChartFontType) {
                        ChartFontType._map = [];
                        ChartFontType._map[0] = "Foreground";
                        ChartFontType.Foreground = 0;
                    })(Charting.ChartFontType || (Charting.ChartFontType = {}));
                    var ChartFontType = Charting.ChartFontType;
                    (function (SeriesRenderType) {
                        SeriesRenderType._map = [];
                        SeriesRenderType.Line = 1;
                        SeriesRenderType.Marks = 2;
                        SeriesRenderType.StepLine = 3;
                    })(Charting.SeriesRenderType || (Charting.SeriesRenderType = {}));
                    var SeriesRenderType = Charting.SeriesRenderType;
                    (function (MarkShapeType) {
                        MarkShapeType._map = [];
                        MarkShapeType.Unknown = 0;
                        MarkShapeType.Circles = 1;
                        MarkShapeType.Triangles = 2;
                        MarkShapeType.VerticalLines = 3;
                        MarkShapeType.Squares = 4;
                    })(Charting.MarkShapeType || (Charting.MarkShapeType = {}));
                    var MarkShapeType = Charting.MarkShapeType;
                    (function (SeriesOrderType) {
                        SeriesOrderType._map = [];
                        SeriesOrderType._map[0] = "Sort";
                        SeriesOrderType.Sort = 0;
                        SeriesOrderType._map[1] = "Remove";
                        SeriesOrderType.Remove = 1;
                    })(Charting.SeriesOrderType || (Charting.SeriesOrderType = {}));
                    var SeriesOrderType = Charting.SeriesOrderType;
                    (function (ChartAreaType) {
                        ChartAreaType._map = [];
                        ChartAreaType._map[0] = "Background";
                        ChartAreaType.Background = 0;
                        ChartAreaType._map[1] = "Foreground";
                        ChartAreaType.Foreground = 1;
                        ChartAreaType._map[2] = "Grid";
                        ChartAreaType.Grid = 2;
                        ChartAreaType._map[3] = "LegendBackground";
                        ChartAreaType.LegendBackground = 3;
                    })(Charting.ChartAreaType || (Charting.ChartAreaType = {}));
                    var ChartAreaType = Charting.ChartAreaType;
                    (function (GraphState) {
                        GraphState._map = [];
                        GraphState.None = 0;
                        GraphState.Roll = 1;
                        GraphState.Stop = 2;
                    })(Charting.GraphState || (Charting.GraphState = {}));
                    var GraphState = Charting.GraphState;
                    var Point2d = (function () {
                        function Point2d(x, y, data, isInView, seriesIndex, xTranslate, yTranslate, xPx, yPx) {
                            if (typeof xTranslate === "undefined") { xTranslate = 0; }
                            if (typeof yTranslate === "undefined") { yTranslate = -1; }
                            this.x = x;
                            this.y = y;
                            this.data = data;
                            this.isInView = isInView;
                            this.seriesIndex = seriesIndex;
                            this.xTranslate = xTranslate;
                            this.yTranslate = yTranslate;
                            this.xPx = xPx;
                            this.yPx = yPx;
                        }
                        return Point2d;
                    })();
                    Charting.Point2d = Point2d;                    
                    var Padding = (function () {
                        function Padding(left, top, right, bottom) {
                            this.left = left;
                            this.top = top;
                            this.right = right;
                            this.bottom = bottom;
                        }
                        return Padding;
                    })();
                    Charting.Padding = Padding;                    
                    var RectangleDimension = (function (_super) {
                        __extends(RectangleDimension, _super);
                        function RectangleDimension(left, top, right, bottom) {
                                                _super.call(this, left, top, right, bottom);
                            if(this.left > this.right || this.top > this.bottom) {
                                throw "Invalid rectange size";
                            }
                        }
                        Object.defineProperty(RectangleDimension.prototype, "width", {
                            get: function () {
                                return this.right - this.left;
                            },
                            enumerable: true,
                            configurable: true
                        });
                        Object.defineProperty(RectangleDimension.prototype, "height", {
                            get: function () {
                                return this.bottom - this.top;
                            },
                            enumerable: true,
                            configurable: true
                        });
                        return RectangleDimension;
                    })(Padding);
                    Charting.RectangleDimension = RectangleDimension;                    
                    var MinMax = (function () {
                        function MinMax(min, max) {
                            this.min = min;
                            this.max = max;
                        }
                        return MinMax;
                    })();
                    Charting.MinMax = MinMax;                    
                    var ChartColorScheme = (function () {
                        function ChartColorScheme(light, dark) {
                            this._light = "#FF0000";
                            this._dark = "#FF0000";
                            this._light = light;
                            this._dark = dark;
                        }
                        Object.defineProperty(ChartColorScheme.prototype, "light", {
                            get: function () {
                                return this._light;
                            },
                            enumerable: true,
                            configurable: true
                        });
                        Object.defineProperty(ChartColorScheme.prototype, "dark", {
                            get: function () {
                                return this._dark;
                            },
                            enumerable: true,
                            configurable: true
                        });
                        return ChartColorScheme;
                    })();
                    Charting.ChartColorScheme = ChartColorScheme;                    
                    var ChartSeries = (function () {
                        function ChartSeries(context, index, legendText, seriesRenderType, counterId, config, markShapeType, colorScheme) {
                            this.colorSchemes = [
                                new ChartColorScheme("rgba(83, 134, 191, 0.50)", "rgb(83, 134, 191)"), 
                                new ChartColorScheme("rgba(167, 148, 50, 0.25)", "rgb(167, 148, 50)"), 
                                
                            ];
                            this.context = context;
                            this.index = index;
                            this.legendText = legendText;
                            this.seriesRenderType = seriesRenderType;
                            this.data = [];
                            this.colorScheme = colorScheme || this.colorSchemes[index];
                            this.counterId = counterId;
                            this.markShapeType = markShapeType;
                            this.graphConfiguration = config;
                        }
                        ChartSeries.prototype.getStartIndexForView = function () {
                            var query = this.graphConfiguration.data.gridX.min;
                            var seriesPoints = this.data;
                            if(!seriesPoints.length) {
                                return -1;
                            }
                            var maxIndex = seriesPoints.length - 1;
                            var minIndex = 0;
                            var pivotIndex;
                            var pivotValue;
                            while(minIndex <= maxIndex) {
                                pivotIndex = Math.floor((minIndex + maxIndex) / 2);
                                pivotValue = seriesPoints[pivotIndex];
                                if(pivotValue.x > query) {
                                    maxIndex = pivotIndex - 1;
                                } else {
                                    if(pivotValue.x < query) {
                                        minIndex = pivotIndex + 1;
                                    } else {
                                        break;
                                    }
                                }
                            }
                            return seriesPoints[pivotIndex].x >= query ? pivotIndex : pivotIndex + 1;
                        };
                        ChartSeries.prototype.removeInvalidPoints = function () {
                            var finalPoints = this.data;
                            if(finalPoints && finalPoints.length > 0 && finalPoints[0].x < this.graphConfiguration.data.dataTimeRangeStart) {
                                var count = 0;
                                for(var i = 0; i <= finalPoints.length; i++) {
                                    if(finalPoints[i].x > this.graphConfiguration.data.dataTimeRangeStart) {
                                        break;
                                    }
                                    count++;
                                }
                                finalPoints.splice(0, count > 0 ? count - 1 : 0);
                            }
                        };
                        ChartSeries.prototype.getPointsToRender = function (chartRect, drawEntireChart) {
                            var renderedPoints = new Array();
                            var seriesPoints = this.data;
                            var seriesIndex = this.index;
                            this.removeInvalidPoints();
                            if(seriesPoints.length > 0) {
                                var startIndex = drawEntireChart ? 0 : seriesPoints.length - 1;
                                var endIndex = seriesPoints.length - 1;
                                var startX = this.graphConfiguration.data.gridX.min;
                                var rangeX = this.graphConfiguration.data.gridRange.x;
                                var endX = this.graphConfiguration.data.gridX.max;
                                var scaleX = chartRect.width / rangeX;
                                if(this.seriesRenderType === SeriesRenderType.Line || this.seriesRenderType === SeriesRenderType.StepLine) {
                                    if(startIndex === 0) {
                                        if(this.seriesRenderType === SeriesRenderType.Line) {
                                            renderedPoints.push(new Point2d(this.graphConfiguration.data.gridX.min, seriesPoints[startIndex].y, null, false, seriesIndex, 0, 0, this.graphConfiguration.style.chartAxisPadding.left, this.getYCoord(seriesPoints[startIndex].y, chartRect.top, chartRect.height)));
                                        } else {
                                            if(this.seriesRenderType === SeriesRenderType.StepLine) {
                                                renderedPoints.push(new Point2d(seriesPoints[startIndex].x, this.graphConfiguration.data.gridY.min, null, false, seriesIndex, 0, 0, chartRect.left + (seriesPoints[startIndex].x - startX) * scaleX, this.getYCoord(this.graphConfiguration.data.gridY.min, chartRect.top, chartRect.height)));
                                            }
                                        }
                                    } else {
                                        startIndex--;
                                    }
                                }
                                var x = 0;
                                var y = 0;
                                var startPoint = seriesPoints[startIndex];
                                var hasMovedToStart = false;
                                for(var i = startIndex; i <= endIndex; i++) {
                                    var point = seriesPoints[i];
                                    var pointInView = (point.x >= startX && point.x <= endX);
                                    var requiresDrawing = pointInView;
                                    point.isInView = pointInView;
                                    point.seriesIndex = seriesIndex;
                                    if(!requiresDrawing) {
                                        if((point.x < startX && i < endIndex && seriesPoints[i + 1].x > startX) || (point.x > endX && i > 0 && seriesPoints[i - 1].x < endX)) {
                                            requiresDrawing = true;
                                        }
                                    }
                                    if(requiresDrawing) {
                                        x = chartRect.left + (point.x - startX) * scaleX;
                                        y = this.getYCoord(point.y, chartRect.top, chartRect.height);
                                        point.xPx = Math.floor(x);
                                        point.yPx = Math.floor(y);
                                        renderedPoints.push(point);
                                    }
                                }
                            }
                            return renderedPoints;
                        };
                        ChartSeries.prototype.getYCoord = function (y, top, height) {
                            return top + ((this.graphConfiguration.data.gridY.max - y) / this.graphConfiguration.data.gridRange.y) * height;
                        };
                        return ChartSeries;
                    })();
                    Charting.ChartSeries = ChartSeries;                    
                    var GraphData = (function () {
                        function GraphData() {
                            this.chartAxisCount = 5;
                            this.chartAxisIncreaseRatio = 1.1;
                            this.scaleForWhiteSpace = 0.05;
                            this.chartDrawFill = true;
                            this.gridX = new MinMax(0, 0);
                            this.gridY = new MinMax(0, 0);
                            this.gridRange = new Point2d(0, 0);
                            this.dataTimeRangeStart = 0;
                            this.graphState = GraphState.None;
                            this.markRerenderTimeRange = 10000;
                        }
                        return GraphData;
                    })();
                    Charting.GraphData = GraphData;                    
                    var GraphStyle = (function () {
                        function GraphStyle() {
                            this.chartAxisPadding = new Padding(15, 15, 5, 15);
                            this.legendPadding = new Padding(10, 5, 5, 5);
                            this.legendSize = 8;
                            this.isLegendHidden = false;
                            this.isAxesHidden = false;
                        }
                        return GraphStyle;
                    })();
                    Charting.GraphStyle = GraphStyle;                    
                    var GraphConfigurations = (function () {
                        function GraphConfigurations(data, style) {
                            if(!data || !style) {
                                throw Error("Either Data or Style are invalid.");
                            }
                            this.data = data;
                            this.style = style;
                        }
                        return GraphConfigurations;
                    })();
                    Charting.GraphConfigurations = GraphConfigurations;                    
                    var Axes = (function () {
                        function Axes(context, config) {
                            this.renderValues = true;
                            this.renderAxes = true;
                            this._context = context;
                            this._chartConfig = config;
                        }
                        Axes.prototype.render = function () {
                            if(this.renderValues === null || this.renderValues === undefined) {
                                this.renderValues = true;
                            }
                            if(this.renderValues) {
                                this.drawChartAxes();
                            }
                            if(this.renderAxes === null || this.renderAxes === undefined) {
                                this.renderAxes = true;
                            }
                            if(this.renderAxes) {
                                this.drawChartBorder();
                            }
                        };
                        Axes.prototype.drawXAxesValue = function (value, offsetY) {
                            this._context.save();
                            this._context.fillStyle = Plugin.Theme.getValue("vs-system-window-text-color");
                            this._context.font = Plugin.Theme.getValue("plugin-font-size") + " " + Plugin.Theme.getValue("plugin-font-family");
                            this._context.fillText(value, this._chartConfig.style.chartAxisPadding.left - 5, this._chartConfig.style.chartAxisPadding.top + offsetY);
                            this._context.restore();
                        };
                        Axes.prototype.drawChartAxes = function () {
                            this._context.fillStyle = Plugin.Theme.getValue("vs-system-window-color");
                            this._context.strokeStyle = Plugin.Theme.getValue("vs-system-window-color");
                            this._context.textAlign = "left";
                            this.drawXAxesValue(this.convertToUnits(this._chartConfig.data.gridY.max, true), -1 * this._chartConfig.style.legendPadding.bottom);
                            this.drawXAxesValue(this.convertToUnits(this._chartConfig.data.gridY.min), this._chartConfig.style.chartHeight - this._chartConfig.style.legendPadding.top);
                        };
                        Axes.prototype.drawChartBorder = function () {
                            this._context.save();
                            this._context.fillStyle = "transparent";
                            this._context.strokeStyle = Plugin.Theme.getValue("vs-file-tab-inactive-text-color");
                            this._context.lineWidth = (this._chartConfig.style.chartRect.height < 100 ? 1 : 2);
                            this._context.beginPath();
                            this._context.moveTo(this._chartConfig.style.chartRect.left - 0.5, this._chartConfig.style.chartRect.top - 0.5);
                            this._context.lineTo(this._chartConfig.style.chartRect.left - 0.5, this._chartConfig.style.chartRect.top - 0.5 + this._chartConfig.style.chartRect.height);
                            this._context.lineTo(this._chartConfig.style.chartRect.left - 0.5 + this._chartConfig.style.chartRect.width, this._chartConfig.style.chartRect.top - 0.5 + this._chartConfig.style.chartRect.height);
                            this._context.stroke();
                            this._context.restore();
                        };
                        Axes.prototype.convertToUnits = function (value, fIncludeUnitsString) {
                            if (typeof fIncludeUnitsString === "undefined") { fIncludeUnitsString = false; }
                            var result = value.toString();
                            for(var i = 0; i < this._chartConfig.data.units.length; i++) {
                                var units = this._chartConfig.data.units[i];
                                if(units.FromValue <= value && value <= units.ToValue) {
                                    if(units.Divider != 1) {
                                        value = value / units.Divider;
                                    }
                                    var decimals = Math.pow(10, units.Decimals);
                                    value = Math.round(value * decimals) / (decimals);
                                    result = value.toFixed(units.Decimals);
                                    if(fIncludeUnitsString) {
                                        result = result + " " + units.Units;
                                    }
                                    break;
                                }
                            }
                            return result;
                        };
                        return Axes;
                    })();
                    Charting.Axes = Axes;                    
                    var Legend = (function () {
                        function Legend(context, config, series) {
                            this._context = context;
                            this._chartConfig = config;
                            this._series = series;
                        }
                        Legend.prototype.render = function () {
                            this._context.save();
                            var legendStartX = 0;
                            for(var i = 0; i < this._series.length; i++) {
                                this._context.fillStyle = Plugin.Theme.getValue("vs-system-window-text-color");
                                this._context.font = Plugin.Theme.getValue("plugin-font-size") + " " + Plugin.Theme.getValue("plugin-font-family");
                                var legendTextLength = this._context.measureText(this._series[i].legendText);
                                this._context.textAlign = "left";
                                legendStartX += legendTextLength.width;
                                this._context.fillText(this._series[i].legendText, this._chartConfig.style.chartWidth - legendStartX, this._chartConfig.style.chartAxisPadding.top - this._chartConfig.style.legendPadding.top);
                                this._context.fillStyle = this._series[i].colorScheme.dark;
                                this._context.strokeStyle = this._series[i].colorScheme.dark;
                                legendStartX += this._chartConfig.style.legendSize + this._chartConfig.style.legendPadding.right;
                                var point = new Point2d(0, 0, null, null, null, 0, 4, this._chartConfig.style.chartWidth - legendStartX, this._chartConfig.style.chartAxisPadding.top - (this._chartConfig.style.legendSize / 2) - this._chartConfig.style.legendPadding.top);
                                if(this._series[i].seriesRenderType === SeriesRenderType.Marks) {
                                    if(this._series[i].markShapeType === MarkShapeType.Circles) {
                                        Mark.drawCircle(point, this._context, this._chartConfig.style.legendSize);
                                    }
                                    if(this._series[i].markShapeType === MarkShapeType.Triangles) {
                                        Mark.drawTriangle(point, this._context, this._chartConfig.style.legendSize);
                                    }
                                    if(this._series[i].markShapeType === MarkShapeType.Squares) {
                                        Mark.drawSquare(point, this._context, this._chartConfig.style.legendSize);
                                    }
                                }
                                if(this._series[i].seriesRenderType === SeriesRenderType.Line || this._series[i].seriesRenderType === SeriesRenderType.StepLine) {
                                    this._context.fillRect(this._chartConfig.style.chartWidth - legendStartX - (this._chartConfig.style.legendSize * 1 / 4), this._chartConfig.style.chartAxisPadding.top - (this._chartConfig.style.legendSize * 3 / 4) - this._chartConfig.style.legendPadding.top, this._chartConfig.style.legendSize, this._chartConfig.style.legendSize / 2);
                                }
                                legendStartX += this._chartConfig.style.legendPadding.left;
                            }
                            this._context.restore();
                        };
                        return Legend;
                    })();
                    Charting.Legend = Legend;                    
                    var Mark = (function () {
                        function Mark() { }
                        Mark.render = function render(markShapeType, point, context, markSize) {
                            if (typeof markSize === "undefined") { markSize = 8; }
                            switch(markShapeType) {
                                case MarkShapeType.Circles: {
                                    Mark.drawCircle(point, context, markSize);
                                    break;

                                }
                                case MarkShapeType.Triangles: {
                                    Mark.drawTriangle(point, context, markSize);
                                    break;

                                }
                                case MarkShapeType.Squares: {
                                    Mark.drawSquare(point, context, markSize);
                                    break;

                                }
                            }
                            ; ;
                        }
                        Mark.drawCircle = function drawCircle(point, context, markSize) {
                            var xt = 0;
                            var yt = 0;
                            if(point.xTranslate) {
                                xt = point.xTranslate;
                            } else {
                                point.xTranslate = xt;
                            }
                            if(point.yTranslate) {
                                yt = point.yTranslate;
                            } else {
                                point.yTranslate = yt;
                            }
                            context.save();
                            context.beginPath();
                            context.translate(point.xPx + xt, point.yPx + yt);
                            context.arc(0, -1 * markSize / 2, markSize / 2, 0, Math.PI * 2);
                            context.fill();
                            context.restore();
                        }
                        Mark.drawTriangle = function drawTriangle(point, context, markSize) {
                            var xt = 0;
                            var yt = 0;
                            if(point.xTranslate) {
                                xt = point.xTranslate;
                            } else {
                                point.xTranslate = xt;
                            }
                            if(point.yTranslate) {
                                yt = point.yTranslate;
                            } else {
                                point.yTranslate = yt;
                            }
                            context.save();
                            context.translate(point.xPx + xt, point.yPx + yt);
                            context.beginPath();
                            context.moveTo(0, -1 * markSize);
                            context.lineTo(-1 * markSize / 2, 0);
                            context.lineTo(markSize / 2, 0);
                            context.closePath();
                            context.fill();
                            context.restore();
                        }
                        Mark.drawSquare = function drawSquare(point, context, markSize) {
                            var xt = 0;
                            var yt = 0;
                            if(point.xTranslate) {
                                xt = point.xTranslate;
                            } else {
                                point.xTranslate = xt;
                            }
                            if(point.yTranslate) {
                                yt = point.yTranslate;
                            } else {
                                point.yTranslate = yt;
                            }
                            context.save();
                            context.translate(point.xPx + xt, point.yPx + yt);
                            context.fillRect(-1 * markSize / 2, -1 * markSize, markSize, markSize);
                            context.restore();
                        }
                        return Mark;
                    })();
                    Charting.Mark = Mark;                    
                    var LineSeries = (function (_super) {
                        __extends(LineSeries, _super);
                        function LineSeries(context, config, index, legendText, seriesRenderType, counterId, markShapeType, colorScheme) {
                                                _super.call(this, context, index, legendText, seriesRenderType, counterId, config, markShapeType, colorScheme);
                        }
                        LineSeries.prototype.render = function (drawEntireChart) {
                            var seriesPoints = this.data;
                            this.removeInvalidPoints();
                            var renderedPoints = this.getPointsToRender(this.graphConfiguration.style.chartRect, drawEntireChart);
                            if(renderedPoints && renderedPoints.length > 0) {
                                this.drawLineFill(renderedPoints, drawEntireChart);
                                this.drawLineSeries(renderedPoints);
                            }
                        };
                        LineSeries.prototype.drawLineFill = function (renderedPoints, drawEntireChart) {
                            if(renderedPoints.length > 0) {
                                var startIndex = 0;
                                var endIndex = renderedPoints.length - 1;
                                var startX = this.graphConfiguration.data.gridX.min;
                                var rangeX = this.graphConfiguration.data.gridRange.x;
                                var endX = this.graphConfiguration.data.gridX.max;
                                var scaleX = this.graphConfiguration.style.chartRect.width / rangeX;
                                var x = 0;
                                var y = 0;
                                var hasMovedToStart = false;
                                this.context.save();
                                this.context.beginPath();
                                var previousPoint = renderedPoints[0];
                                this.context.rect(this.graphConfiguration.style.chartRect.left, this.graphConfiguration.style.chartRect.top, this.graphConfiguration.style.chartRect.width, this.graphConfiguration.style.chartRect.height);
                                this.context.clip();
                                this.context.fillStyle = Plugin.Theme.getValue("vs-system-window-color");
                                var xPos = previousPoint.x >= this.graphConfiguration.data.gridX.min ? previousPoint.xPx : this.graphConfiguration.style.chartAxisPadding.left;
                                this.context.fillRect(xPos, this.graphConfiguration.style.chartRect.top, this.graphConfiguration.style.chartWidth - xPos, this.graphConfiguration.style.chartRect.height);
                                this.context.fillStyle = this.colorScheme.light;
                                this.context.strokeStyle = this.colorScheme.dark;
                                this.context.beginPath();
                                for(var i = startIndex; i <= endIndex; i++) {
                                    var point = renderedPoints[i];
                                    if(!hasMovedToStart) {
                                        this.context.moveTo(point.xPx, point.yPx);
                                        hasMovedToStart = true;
                                    } else {
                                        this.context.lineTo(point.xPx, point.yPx);
                                    }
                                    previousPoint = point;
                                }
                                if(this.graphConfiguration.data.chartDrawFill) {
                                    var y = this.getYCoord(0, this.graphConfiguration.style.chartRect.top, this.graphConfiguration.style.chartRect.height);
                                    this.context.lineTo(point.xPx, y);
                                    this.context.lineTo(drawEntireChart ? this.graphConfiguration.style.chartAxisPadding.left : renderedPoints[0].xPx, y);
                                    this.context.closePath();
                                    this.context.fill();
                                } else {
                                    this.context.stroke();
                                }
                                this.context.restore();
                            }
                        };
                        LineSeries.prototype.drawLineSeries = function (renderedPoints) {
                            if(renderedPoints.length > 0) {
                                this.context.save();
                                this.context.beginPath();
                                if(this.graphConfiguration.data.chartDrawFill) {
                                    this.context.lineWidth = (this.graphConfiguration.style.chartRect.height < 100 ? 1 : 2);
                                    this.context.strokeStyle = this.colorScheme.dark;
                                    this.context.beginPath();
                                    for(var i = 0; i < renderedPoints.length; i++) {
                                        var rp = renderedPoints[i];
                                        if(i === 0) {
                                            this.context.moveTo(renderedPoints[i].xPx, renderedPoints[i].yPx - 0.5);
                                        } else {
                                            this.context.lineTo(rp.xPx, Math.round(rp.yPx) - 0.5);
                                        }
                                    }
                                    this.context.stroke();
                                }
                                this.context.restore();
                            }
                        };
                        return LineSeries;
                    })(ChartSeries);
                    Charting.LineSeries = LineSeries;                    
                    var StepLineSeries = (function (_super) {
                        __extends(StepLineSeries, _super);
                        function StepLineSeries(context, config, index, legendText, seriesRenderType, counterId, markShapeType, colorScheme) {
                                                _super.call(this, context, index, legendText, seriesRenderType, counterId, config, markShapeType, colorScheme);
                        }
                        StepLineSeries.prototype.render = function (drawEntireChart) {
                            var seriesPoints = this.data;
                            this.removeInvalidPoints();
                            var renderedPoints = this.getPointsToRender(this.graphConfiguration.style.chartRect, drawEntireChart);
                            if(renderedPoints && renderedPoints.length > 0) {
                                this.drawLineFill(renderedPoints, drawEntireChart);
                                this.drawLineSeries(renderedPoints);
                            }
                        };
                        StepLineSeries.prototype.drawLineFill = function (renderedPoints, drawEntireChart) {
                            if(renderedPoints.length > 0) {
                                var startIndex = 0;
                                var endIndex = renderedPoints.length - 1;
                                var startX = this.graphConfiguration.data.gridX.min;
                                var rangeX = this.graphConfiguration.data.gridRange.x;
                                var endX = this.graphConfiguration.data.gridX.max;
                                var scaleX = this.graphConfiguration.style.chartRect.width / rangeX;
                                var x = 0;
                                var y = 0;
                                var hasMovedToStart = false;
                                this.context.save();
                                this.context.beginPath();
                                var previousPoint = renderedPoints[0];
                                this.context.rect(this.graphConfiguration.style.chartRect.left, this.graphConfiguration.style.chartRect.top, this.graphConfiguration.style.chartRect.width, this.graphConfiguration.style.chartRect.height);
                                this.context.clip();
                                this.context.fillStyle = Plugin.Theme.getValue("vs-system-window-color");
                                var xPos = previousPoint.x >= this.graphConfiguration.data.gridX.min ? previousPoint.xPx : this.graphConfiguration.style.chartAxisPadding.left;
                                this.context.fillRect(xPos, this.graphConfiguration.style.chartRect.top, this.graphConfiguration.style.chartWidth - xPos, this.graphConfiguration.style.chartRect.height);
                                this.context.fillStyle = this.colorScheme.light;
                                this.context.strokeStyle = this.colorScheme.dark;
                                this.context.beginPath();
                                for(var i = startIndex; i <= endIndex; i++) {
                                    var point = renderedPoints[i];
                                    if(!hasMovedToStart) {
                                        this.context.moveTo(point.xPx, point.yPx);
                                        hasMovedToStart = true;
                                    } else {
                                        this.context.lineTo(point.xPx, previousPoint.yPx);
                                        this.context.lineTo(point.xPx, point.yPx);
                                    }
                                    previousPoint = point;
                                }
                                if(this.graphConfiguration.data.chartDrawFill) {
                                    var y = this.getYCoord(0, this.graphConfiguration.style.chartRect.top, this.graphConfiguration.style.chartRect.height);
                                    this.context.lineTo(point.xPx, y);
                                    this.context.lineTo(drawEntireChart ? this.graphConfiguration.style.chartAxisPadding.left : renderedPoints[0].xPx, y);
                                    this.context.closePath();
                                    this.context.fill();
                                } else {
                                    this.context.stroke();
                                }
                                this.context.restore();
                            }
                        };
                        StepLineSeries.prototype.drawLineSeries = function (renderedPoints) {
                            if(renderedPoints.length > 0) {
                                this.context.save();
                                this.context.beginPath();
                                if(this.graphConfiguration.data.chartDrawFill) {
                                    this.context.lineWidth = (this.graphConfiguration.style.chartRect.height < 100 ? 1 : 2);
                                    this.context.strokeStyle = this.colorScheme.dark;
                                    this.context.beginPath();
                                    var previousPoint = null;
                                    for(var i = 0; i < renderedPoints.length; i++) {
                                        var rp = renderedPoints[i];
                                        if(i === 0) {
                                            this.context.moveTo(renderedPoints[i].xPx, renderedPoints[i].yPx - 0.5);
                                        } else {
                                            var yPos = this.graphConfiguration.style.chartHeight - 0.5;
                                            if(previousPoint && previousPoint.yPx) {
                                                yPos = Math.round(previousPoint.yPx) - 0.5;
                                            }
                                            this.context.lineTo(rp.xPx, yPos);
                                            this.context.lineTo(rp.xPx, Math.round(rp.yPx) - 0.5);
                                        }
                                        previousPoint = rp;
                                    }
                                    this.context.stroke();
                                }
                                this.context.restore();
                            }
                        };
                        return StepLineSeries;
                    })(ChartSeries);
                    Charting.StepLineSeries = StepLineSeries;                    
                    var MarkSeries = (function (_super) {
                        __extends(MarkSeries, _super);
                        function MarkSeries(context, config, index, legendText, seriesRenderType, counterId, markShapeType, colorScheme) {
                                                _super.call(this, context, index, legendText, seriesRenderType, counterId, config, markShapeType, colorScheme);
                        }
                        MarkSeries.prototype.drawCircles = function (renderedPoints) {
                            for(var i = 0; i < renderedPoints.length; i++) {
                                if(renderedPoints[i].isInView) {
                                    Mark.drawCircle(renderedPoints[i], this.context, this.graphConfiguration.style.legendSize);
                                }
                            }
                        };
                        MarkSeries.prototype.drawTriangles = function (renderedPoints) {
                            for(var i = 0; i < renderedPoints.length; i++) {
                                if(renderedPoints[i].isInView) {
                                    Mark.drawTriangle(renderedPoints[i], this.context, this.graphConfiguration.style.legendSize);
                                }
                            }
                        };
                        MarkSeries.prototype.drawSquares = function (renderedPoints) {
                            for(var i = 0; i < renderedPoints.length; i++) {
                                if(renderedPoints[i].isInView) {
                                    Mark.drawSquare(renderedPoints[i], this.context, this.graphConfiguration.style.legendSize);
                                }
                            }
                        };
                        MarkSeries.prototype.render = function (drawEntireChart) {
                            var seriesPoints = this.data;
                            this.removeInvalidPoints();
                            var renderedPoints = this.getPointsToRender(this.graphConfiguration.style.chartRect, drawEntireChart);
                            if(drawEntireChart && (!renderedPoints || renderedPoints.length <= 0)) {
                                return;
                            }
                            var seriesIndexForColor = -1;
                            var seriesPoints = this.data;
                            if(!drawEntireChart && this.referencePoint) {
                                renderedPoints = [];
                                for(var i = seriesPoints.length - 1; 0 <= i; i--) {
                                    var point = seriesPoints[i];
                                    if(point.x <= this.referencePoint.x && point.x > this.referencePoint.x - this.graphConfiguration.data.markRerenderTimeRange && point.x >= this.graphConfiguration.data.gridX.min) {
                                        renderedPoints.push(point);
                                    } else {
                                        break;
                                    }
                                }
                            }
                            this.context.fillStyle = this.colorScheme.dark;
                            this.context.strokeStyle = this.colorScheme.dark;
                            switch(this.markShapeType) {
                                case MarkShapeType.Circles: {
                                    this.drawCircles(renderedPoints);
                                    break;

                                }
                                case MarkShapeType.Triangles: {
                                    this.drawTriangles(renderedPoints);
                                    break;

                                }
                                case MarkShapeType.Squares: {
                                    this.drawSquares(renderedPoints);
                                    break;

                                }
                            }
                        };
                        return MarkSeries;
                    })(ChartSeries);
                    Charting.MarkSeries = MarkSeries;                    
                    var Graph = (function () {
                        function Graph(config) {
                            this._series = [];
                            this._containerId = config.containerId;
                            this._container = document.getElementById(this._containerId);
                            if(this._container !== null) {
                                while(this._container.childNodes.length > 0) {
                                    this._container.removeChild(this._container.firstChild);
                                }
                                this._canvas = document.createElement("canvas");
                                this._container.appendChild(this._canvas);
                                this._canvas.style.background = Plugin.Theme.getValue("vs-system-window-color");
                                if(!this._container.runtimeStyle.position || this._container.runtimeStyle.position === "static") {
                                    this._container.style.position = "relative";
                                }
                                this._canvas.addEventListener('mousemove', this.onMouseOver.bind(this));
                                this._canvas.addEventListener('mouseout', this.onMouseOut.bind(this));
                                this._context = this._canvas.getContext("2d");
                                this._config = config;
                                this.initialize();
                                this._axes = new Axes(this._context, this._graphConfiguration);
                                this._legend = new Legend(this._context, this._graphConfiguration, this._series);
                                window.addEventListener("resize", this.onResize.bind(this));
                                this.onResize(null);
                                this._graphConfiguration.data.graphState = GraphState.Roll;
                                Plugin.Theme.addEventListener("themechanged", this.onThemeChanged.bind(this));
                            } else {
                                throw "An element with id " + this._containerId + " could not be found";
                            }
                        }
                        Graph.prototype.onThemeChanged = function (args) {
                            this.render(true);
                        };
                        Graph.prototype.initialize = function () {
                            this._graphConfiguration = new GraphConfigurations(new GraphData(), new GraphStyle());
                            this._graphConfiguration.style.isAxesHidden = (this._config.isAxesHidden != null || this._config.isAxesHidden != undefined) ? this._config.isAxesHidden : this._graphConfiguration.style.isAxesHidden;
                            if(this._graphConfiguration.style.isAxesHidden) {
                                this._graphConfiguration.style.chartAxisPadding = new Padding(0, 15, 15, 0);
                            }
                            this._graphConfiguration.style.isLegendHidden = (this._config.isLegendHidden != null || this._config.isLegendHidden != undefined) ? this._config.isLegendHidden : this._graphConfiguration.style.isLegendHidden;
                            if(this._config.logger) {
                                this._graphConfiguration.data.logger = this._config.logger;
                            } else {
                                this._graphConfiguration.data.logger = Microsoft.VisualStudio.DiagnosticsHub.getLogger();
                            }
                            if(this._config.timeProperties) {
                                this._graphConfiguration.data.timeProperties = new QpcTimeProperties(this._config.timeProperties.CollectionStartTime, this._config.timeProperties.Frequency);
                            } else {
                                this._graphConfiguration.data.timeProperties = new QpcTimeProperties(0, 1000);
                            }
                            if(this._config.windowsZoomLevel) {
                                this._graphConfiguration.style.windowsZoomLevel = this._config.windowsZoomLevel;
                            } else {
                                this._graphConfiguration.style.windowsZoomLevel = {
                                    X: 1,
                                    Y: 1
                                };
                            }
                            this._graphConfiguration.style.chartAxisPadding = new Padding(this._graphConfiguration.style.chartAxisPadding.left * this._graphConfiguration.style.windowsZoomLevel.X, this._graphConfiguration.style.chartAxisPadding.top * this._graphConfiguration.style.windowsZoomLevel.Y, this._graphConfiguration.style.chartAxisPadding.right * this._graphConfiguration.style.windowsZoomLevel.X, this._graphConfiguration.style.chartAxisPadding.bottom * this._graphConfiguration.style.windowsZoomLevel.Y);
                            if(this._config.data.Units) {
                                this._graphConfiguration.data.units = this._config.data.Units;
                            }
                            this._graphConfiguration.data.gridY.min = this._config.data.MinValue;
                            this._graphConfiguration.data.gridY.max = this._config.data.MaxValue;
                            this._graphConfiguration.data.gridRange.y = this._graphConfiguration.data.gridY.max - this._graphConfiguration.data.gridY.min;
                            if(this._config.timeRange) {
                                this._graphConfiguration.data.gridX.min = this._config.timeRange.ViewportTimeRange.Start;
                                this._graphConfiguration.data.gridX.max = this._config.timeRange.ViewportTimeRange.End;
                                this._graphConfiguration.data.gridRange.x = this._graphConfiguration.data.gridX.max - this._graphConfiguration.data.gridX.min;
                            }
                            for(var i = 0; i < this._config.data.Series.length; i++) {
                                this.addSeries(this._config.data.Series[i].Legend, this._config.data.Series[i].SeriesType, this._config.data.Series[i].DataSource.CounterId, this._config.data.Series[i].MarkShapeType);
                            }
                        };
                        Graph.prototype.onResize = function (evt) {
                            if(this.calculateChartSize()) {
                                this.render(true);
                                if(this._mousePosition) {
                                    this.onMouseOver({
                                        mousePosition: this._mousePosition
                                    });
                                }
                            }
                        };
                        Graph.prototype.calculateChartSize = function () {
                            var isCalculated = false;
                            if(this._container.clientWidth > 0 && this._container.clientHeight > 0) {
                                this._canvas.width = this._container.clientWidth;
                                this._canvas.height = this._container.clientHeight;
                                this._graphConfiguration.style.chartWidth = this._canvas.width - this._graphConfiguration.style.chartAxisPadding.right;
                                this._graphConfiguration.style.chartHeight = this._canvas.height - this._graphConfiguration.style.chartAxisPadding.bottom;
                                isCalculated = true;
                            }
                            return isCalculated;
                        };
                        Graph.prototype.renderSeries = function (drawEntireChart) {
                            var renderedPoints = new Array();
                            var lastPoint;
                            for(var seriesIndex = 0; seriesIndex < this._series.length; seriesIndex++) {
                                var series = this._series[seriesIndex];
                                series.referencePoint = this.lastPointOfEntireSeries;
                                series.render(drawEntireChart);
                            }
                            this._context.save();
                            this._context.fillStyle = Plugin.Theme.getValue("vs-system-window-color");
                            this._context.fillRect(0, this._graphConfiguration.style.chartRect.top, this._graphConfiguration.style.chartAxisPadding.left, this._graphConfiguration.style.chartRect.height);
                            this._context.restore();
                        };
                        Graph.prototype.render = function (drawEntireChart) {
                            if(this._graphConfiguration.style.chartAxisPadding.left <= this._graphConfiguration.style.chartWidth && this._graphConfiguration.style.chartAxisPadding.top <= this._graphConfiguration.style.chartHeight) {
                                if(drawEntireChart) {
                                    this.drawChartBackground();
                                }
                                this._graphConfiguration.style.chartRect = new RectangleDimension(this._graphConfiguration.style.chartAxisPadding.left, this._graphConfiguration.style.chartAxisPadding.top, this._graphConfiguration.style.chartWidth, this._graphConfiguration.style.chartHeight);
                                if(!this._graphConfiguration.style.isLegendHidden && drawEntireChart) {
                                    this._legend.render();
                                }
                                if(this._graphConfiguration.data.graphState === GraphState.Roll) {
                                    this.renderSeries(drawEntireChart);
                                }
                                if(!this._graphConfiguration.style.isAxesHidden) {
                                    this._axes.renderValues = this._graphConfiguration.data.units && this._graphConfiguration.data.units.length > 0 && drawEntireChart ? true : false;
                                    this._axes.render();
                                }
                            }
                        };
                        Graph.prototype.drawChartBackground = function () {
                            this._context.fillStyle = Plugin.Theme.getValue("vs-system-window-color");
                            this._context.fillRect(0, 0, this._canvas.width, this._canvas.height);
                        };
                        Graph.prototype.calculateYRange = function (point) {
                            var drawEntireChart = false;
                            if(this._graphConfiguration.data.gridY.min > point.y) {
                                this._graphConfiguration.data.gridY.min = Math.floor((point.y * this._graphConfiguration.data.chartAxisIncreaseRatio) / this._graphConfiguration.data.chartAxisCount) * this._graphConfiguration.data.chartAxisCount;
                            }
                            if(this._graphConfiguration.data.gridY.max < point.y + Math.ceil(this._graphConfiguration.data.gridY.max * this._graphConfiguration.data.scaleForWhiteSpace)) {
                                this._graphConfiguration.data.gridY.max = Math.ceil((point.y * this._graphConfiguration.data.chartAxisIncreaseRatio) / this._graphConfiguration.data.chartAxisCount) * this._graphConfiguration.data.chartAxisCount;
                                drawEntireChart = true;
                            }
                            this._graphConfiguration.data.gridRange.y = this._graphConfiguration.data.gridY.max - this._graphConfiguration.data.gridY.min;
                            return drawEntireChart;
                        };
                        Graph.prototype.addSeries = function (legendText, seriesRenderType, counterId, markShapeType) {
                            var newSeries;
                            switch(seriesRenderType) {
                                case SeriesRenderType.Line: {
                                    newSeries = new LineSeries(this._context, this._graphConfiguration, this._series.length, legendText, seriesRenderType, counterId, markShapeType);
                                    break;

                                }
                                case SeriesRenderType.StepLine: {
                                    newSeries = new StepLineSeries(this._context, this._graphConfiguration, this._series.length, legendText, seriesRenderType, counterId, markShapeType);
                                    break;

                                }
                                case SeriesRenderType.Marks: {
                                    newSeries = new MarkSeries(this._context, this._graphConfiguration, this._series.length, legendText, seriesRenderType, counterId, markShapeType);
                                    break;

                                }
                            }
                            ; ;
                            if(!newSeries) {
                                return;
                            }
                            this._series.push(newSeries);
                            var filledLines = 0;
                            for(var i = 0; i < this._series.length; i++) {
                                if(this._series[i].seriesRenderType === SeriesRenderType.Line || this._series[i].seriesRenderType === SeriesRenderType.StepLine) {
                                    filledLines++;
                                    if(filledLines >= 2) {
                                        this._graphConfiguration.data.chartDrawFill = false;
                                        break;
                                    }
                                }
                            }
                        };
                        Graph.prototype.drawToolTip = function (points, mousePosition) {
                            if(!points || points.length === 0) {
                                return;
                            }
                            var pointsWithMessages = [];
                            for(var i = 0; i < points.length; i++) {
                                if(points[i].data) {
                                    pointsWithMessages.push(points[i].data);
                                }
                            }
                            var message = "";
                            var seperator = "<div class='line-separator'></div>";
                            var count = pointsWithMessages.length - 1;
                            while(count >= 0) {
                                message = pointsWithMessages[count] + message;
                                if(count > 0) {
                                    message = seperator + message;
                                }
                                count--;
                            }
                            if(message) {
                                this._graphConfiguration.data.logger.debug("x: " + mousePosition.xTranslate + ", y: " + mousePosition.yTranslate);
                                var config = {
                                    content: message,
                                    resource: message,
                                    delay: 0
                                };
                                Plugin.VS.Tooltip.show(config);
                            }
                        };
                        Graph.prototype.getMousePosition = function (evt) {
                            var rect = this._canvas.getBoundingClientRect();
                            var root = document.documentElement;
                            var mouseX = evt.clientX - rect.top - root.scrollTop;
                            var mouseY = evt.clientY - rect.left - root.scrollLeft;
                            return new Point2d(mouseX, mouseY, null, null, null, evt.screenX, evt.screenY);
                        };
                        Graph.prototype.onMouseOver = function (evt) {
                            var mousePos = null;
                            if(evt.mousePosition) {
                                mousePos = evt.mousePosition;
                            } else {
                                mousePos = this.getMousePosition(evt);
                            }
                            this._mousePosition = mousePos;
                            this.onMouseOut(null);
                            var count = 0;
                            if(mousePos.x >= this._graphConfiguration.style.chartAxisPadding.left && mousePos.x <= this._graphConfiguration.style.chartWidth && mousePos.y >= this._graphConfiguration.style.chartAxisPadding.top && mousePos.y <= this._graphConfiguration.style.chartHeight) {
                                var foundPoints = this.getMousePointInsideShape(mousePos);
                                if(!foundPoints || foundPoints.length === 0) {
                                    this._foundPoints = null;
                                    Plugin.VS.Tooltip.dismiss();
                                } else {
                                    this.drawCursor(mousePos);
                                }
                                if(this._foundPoints && foundPoints && this._foundPoints.length === foundPoints.length) {
                                    for(var i = 0; i < foundPoints.length; i++) {
                                        if(this._foundPoints[i].x === foundPoints[i].x && this._foundPoints[i].xPx === foundPoints[i].xPx && this._foundPoints[i].y === foundPoints[i].y && this._foundPoints[i].yPx === foundPoints[i].yPx) {
                                            count++;
                                        }
                                    }
                                }
                                if(count !== foundPoints.length && foundPoints.length > 0) {
                                    this._foundPoints = foundPoints;
                                    this.drawToolTip(foundPoints, mousePos);
                                }
                            }
                        };
                        Graph.prototype.onMouseOut = function (evt) {
                            var mDiv = document.getElementById("mDiv");
                            if(mDiv) {
                                mDiv.style.visibility = "hidden";
                            }
                            if(this._mousePosition && evt) {
                                this._mousePosition = null;
                            }
                        };
                        Graph.prototype.drawCursor = function (point) {
                            var mDiv = document.getElementById("mDiv");
                            if(!mDiv) {
                                mDiv = document.createElement("div");
                                mDiv.id = "mDiv";
                                mDiv.style.position = "absolute";
                                mDiv.style.left = point.x.toString() + "px";
                                mDiv.style.top = this._graphConfiguration.style.chartAxisPadding.top.toString() + "px";
                                mDiv.style.height = (this._graphConfiguration.style.chartHeight - this._graphConfiguration.style.chartAxisPadding.top).toString() + "px";
                                mDiv.style.width = "1px";
                                mDiv.style.zIndex = "100";
                                mDiv.style.backgroundColor = Plugin.Theme.getValue("vs-file-tab-inactive-text-color");
                                mDiv.style.visibility = "visible";
                                this._container.appendChild(mDiv);
                            } else {
                                mDiv.style.left = point.x.toString() + "px";
                                mDiv.style.top = this._graphConfiguration.style.chartAxisPadding.top.toString() + "px";
                                mDiv.style.visibility = "visible";
                            }
                        };
                        Graph.prototype.getMousePointInsideShape = function (point) {
                            var foundPoints = [];
                            for(var i = 0; i < this._series.length; i++) {
                                if(this._series[i].seriesRenderType === SeriesRenderType.Marks) {
                                    var seriesData = this._series[i].data;
                                    for(var j = 0; j < seriesData.length; j++) {
                                        var renderedPoint = seriesData[j];
                                        if(renderedPoint.x >= this._graphConfiguration.data.gridX.min && renderedPoint.x <= this._graphConfiguration.data.gridX.max && renderedPoint.xPx && point.x >= renderedPoint.xPx - (this._graphConfiguration.style.legendSize / 2) - 1 && point.x < renderedPoint.xPx + (this._graphConfiguration.style.legendSize / 2) + 1 && point.y <= this._graphConfiguration.style.chartHeight && point.y >= this._graphConfiguration.style.chartAxisPadding.top) {
                                            foundPoints.push(renderedPoint);
                                        }
                                    }
                                }
                            }
                            return foundPoints;
                        };
                        Object.defineProperty(Graph.prototype, "lastPointOfEntireSeries", {
                            get: function () {
                                var validSeries = [];
                                var min = null;
                                for(var i = 0; i < this._series.length; i++) {
                                    if(this._series[i].data && this._series[i].data.length > 0) {
                                        var temp = this._series[i].data[this._series[i].data.length - 1];
                                        if(min) {
                                            if(min.x > temp.x) {
                                                min = temp;
                                            }
                                        } else {
                                            min = temp;
                                        }
                                    }
                                }
                                return min;
                            },
                            enumerable: true,
                            configurable: true
                        });
                        Graph.prototype.setTimeRange = function (timeRange) {
                            var drawEntireChart = false;
                            if(this._graphConfiguration.data.gridX.max != timeRange.ViewportTimeRange.End || this._graphConfiguration.data.gridX.min != timeRange.ViewportTimeRange.Start) {
                                drawEntireChart = true;
                                this.calculateChartSize();
                                this._graphConfiguration.data.gridX.min = timeRange.ViewportTimeRange.Start;
                                this._graphConfiguration.data.gridX.max = timeRange.ViewportTimeRange.End;
                                this._graphConfiguration.data.gridRange.x = this._graphConfiguration.data.gridX.max - this._graphConfiguration.data.gridX.min;
                            }
                            this._graphConfiguration.data.dataTimeRangeStart = timeRange.DataTimeRange.Start;
                            if(this._graphConfiguration.data.graphState === GraphState.Roll) {
                                var seriesIndex = -1;
                                for(var i = 0; i < this._series.length; i++) {
                                    if(this._series[i].seriesRenderType === SeriesRenderType.StepLine) {
                                        seriesIndex = i;
                                        break;
                                    }
                                }
                                if(seriesIndex >= 0 && this._series[seriesIndex].seriesRenderType === SeriesRenderType.StepLine && timeRange.DataTimeRange.End) {
                                    var y = this._graphConfiguration.style.chartHeight;
                                    if(this._series[seriesIndex].data && this._series[seriesIndex].data.length > 0) {
                                        y = this._series[seriesIndex].data[this._series[seriesIndex].data.length - 1].y;
                                    }
                                    this._series[seriesIndex].data.push(new Point2d(timeRange.DataTimeRange.End, y));
                                    this.render(drawEntireChart);
                                    this._series[seriesIndex].data.pop();
                                } else {
                                    this.render(drawEntireChart);
                                }
                                if(this._mousePosition) {
                                    this.onMouseOver({
                                        mousePosition: this._mousePosition
                                    });
                                }
                            }
                        };
                        Graph.prototype.setGraphState = function (graphState) {
                            this._graphConfiguration.data.graphState = graphState;
                        };
                        Graph.prototype.addPointToSeries = function (counterId, timestamp, value, tooltip) {
                            var milliseconds = this._graphConfiguration.data.timeProperties.convertQpcTimestampToMilliseconds(timestamp);
                            var seriesData;
                            for(var i = 0; i < this._series.length; i++) {
                                if(this._series[i].counterId === counterId) {
                                    seriesData = this._series[i].data;
                                }
                            }
                            if(!seriesData) {
                                throw Error("Invalid counterId");
                            }
                            if(seriesData.length > 0) {
                                if(seriesData[seriesData.length - 1].x >= milliseconds) {
                                    return;
                                }
                            }
                            var newPoint = new Point2d(milliseconds, value, tooltip);
                            seriesData.push(newPoint);
                            var drawEntireChart = this.calculateYRange(newPoint);
                            if(this._graphConfiguration.data.graphState === GraphState.Roll) {
                                this.render(drawEntireChart);
                                if(this._mousePosition) {
                                    this.onMouseOver({
                                        mousePosition: this._mousePosition
                                    });
                                }
                            }
                        };
                        return Graph;
                    })();
                    Charting.Graph = Graph;                    
                })(Controls.Charting || (Controls.Charting = {}));
                var Charting = Controls.Charting;
            })(DiagnosticsHub.Controls || (DiagnosticsHub.Controls = {}));
            var Controls = DiagnosticsHub.Controls;
        })(VisualStudio.DiagnosticsHub || (VisualStudio.DiagnosticsHub = {}));
        var DiagnosticsHub = VisualStudio.DiagnosticsHub;
    })(Microsoft.VisualStudio || (Microsoft.VisualStudio = {}));
    var VisualStudio = Microsoft.VisualStudio;
})(Microsoft || (Microsoft = {}));
//@ sourceMappingURL=Controls.js.map

// SIG // Begin signature block
// SIG // MIIajQYJKoZIhvcNAQcCoIIafjCCGnoCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFL4R5MsgvzwT
// SIG // Wc3qunHUh0S2YCTHoIIVeTCCBLowggOioAMCAQICCmEC
// SIG // jkIAAAAAAB8wDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OFoXDTEz
// SIG // MDQwOTIyMjU1OFowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpGNTI4LTM3NzctOEE3NjElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAJbsjkdN
// SIG // VMJclYDXTgs9v5dDw0vjYGcRLwFNDNjRRi8QQN4LpFBS
// SIG // EogLQ3otP+5IbmbHkeYDym7sealqI5vNYp7NaqQ/56ND
// SIG // /2JHobS6RPrfQMGFVH7ooKcsQyObUh8yNfT+mlafjWN3
// SIG // ezCeCjOFchvKSsjMJc3bXREux7CM8Y9DSEcFtXogC+Xz
// SIG // 78G69LPYzTiP+yGqPQpthRfQyueGA8Azg7UlxMxanMTD
// SIG // 2mIlTVMlFGGP+xvg7PdHxoBF5jVTIzZ3yrDdmCs5wHU1
// SIG // D92BTCE9djDFsrBlcylIJ9jC0rCER7t4utV0A97XSxn3
// SIG // U9542ob3YYgmM7RHxqBUiBUrLHUCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBQv6EbIaNNuT7Ig0N6JTvFH7kjB8jAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBz/30unc2NiCt8feNeFXHp
// SIG // aGLwCLZDVsRcSi1o2PlIEZHzEZyF7BLUVKB1qTihWX91
// SIG // 7sb1NNhUpOLQzHyXq5N1MJcHHQRTLDZ/f/FAHgybgOIS
// SIG // CiA6McAHdWfg+jSc7Ij7VxzlWGIgkEUvXUWpyI6zfHJt
// SIG // ECfFS9hvoqgSs201I2f6LNslLbldsR4F50MoPpwFdnfx
// SIG // Jd4FRxlt3kmFodpKSwhGITWodTZMt7MIqt+3K9m+Kmr9
// SIG // 3zUXzD8Mx90Gz06UJGMgCy4krl9DRBJ6XN0326RFs5E6
// SIG // Eld940fGZtPPnEZW9EwHseAMqtX21Tyi4LXU+Bx+BFUQ
// SIG // axj0kc1Rp5VlMIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggSAMIIEfAIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIGiMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBQslxKIG+SOQILcAZUhl4Zy
// SIG // M42GLzBCBgorBgEEAYI3AgEMMTQwMqAYgBYAQwBvAG4A
// SIG // dAByAG8AbABzAC4AagBzoRaAFGh0dHA6Ly9taWNyb3Nv
// SIG // ZnQuY29tMA0GCSqGSIb3DQEBAQUABIIBAEg953AD0e8c
// SIG // cjHRZX4LRFhxIWvIGGL1BcxbyJpiqOo433CfdseTYEe9
// SIG // 6gff/dk0+SDd3AzURT+zRSoSDsAGpUwLOKqsTSkOMh8b
// SIG // 2xsbkA1XwjQ94WIhFz17ZY3UQMiZn1i25xjwyb6kWzES
// SIG // UwpEH+KUvDPzZYtGoZhGUcOde9ZaXIle8Qg6+BTnfkmK
// SIG // F9Zt1yItY4Qy59Az4fA/Bs9DaCr1PPY+c+uAXRDNMBoo
// SIG // a3ZRsrcwluTZZPAKvmp9aPr6VG2qpr46RcUlhYncLQIC
// SIG // FRmj9KQoped4XFkRaMwd7h+OSh1+F+PKmACzIQfJYFdu
// SIG // vyFtkTnOuTApGmTNLVWmgOmhggIfMIICGwYJKoZIhvcN
// SIG // AQkGMYICDDCCAggCAQEwgYUwdzELMAkGA1UEBhMCVVMx
// SIG // EzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1Jl
// SIG // ZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
// SIG // dGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRpbWUtU3Rh
// SIG // bXAgUENBAgphAo5CAAAAAAAfMAkGBSsOAwIaBQCgXTAY
// SIG // BgkqhkiG9w0BCQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3
// SIG // DQEJBTEPFw0xMzAzMTUwNjM0MDJaMCMGCSqGSIb3DQEJ
// SIG // BDEWBBSSHIC2a8VOvOugJxaxByrKRgbeEDANBgkqhkiG
// SIG // 9w0BAQUFAASCAQCBJmQV4lAMAQOalRE4VTDOMdwYOK0i
// SIG // CdJ/8infUU2ml58giLLBqjQvVbctezp06E0/q5M+B26D
// SIG // 9DyQ0pzaqODK3rCaTIC/4cDNy3eUqZhXHcPVVWlUNPgU
// SIG // teUkyd2KK6GOsHEutVfUnlYKXOR5oNW/c9AJuuzRJTc9
// SIG // uy2HDrX+8r9/Rh7VzMCoC9CpkzRriPChnsHZlY+OP3mV
// SIG // MS4vk75w3c1VaAl7bCIwoPdX/ZNf7M/uFsMwFuHNQI7F
// SIG // 7LZeO69A3EY9eE94B+jbW1p9tApuD6Scaagfg40UMoT6
// SIG // 28xjQoO8Pabo66I4ydPJFyQ+87L9lQ53SjniPxBXvk3R
// SIG // h3By
// SIG // End signature block
